﻿using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using PpmsDataWebService.ModelsEnumTypes;
using PpmsDataWebService.Models;
using VA.PPMS.Context;

namespace PpmsDataWebService.Mappers
{
    public class FacilityMap
    {
        public static async Task<IQueryable<Facility>> MapFacilities(List<va_facility> ppmsFacilities, string mapRelatedType, PpmsContext context)
        {
            var facilities = new List<Facility> { };
            foreach (var ppmsFacility in ppmsFacilities)
            {
                var facility = new Facility
                {
                    Name = ppmsFacility.va_name,
                    StationName = ppmsFacility.ppms_StationName,
                    StationNumber = ppmsFacility.va_stationnumber,
                    FacilityName = ppmsFacility.ppms_Facility,
                    Address1 = ppmsFacility.ppms_Address1,
                    Address2 = ppmsFacility.ppms_Address2,
                    Address3 = ppmsFacility.ppms_Address3,
                    City = ppmsFacility.va_city,
                    State = ppmsFacility.va_state,
                    ZipCode = ppmsFacility.va_zipcode
                };

                if (ppmsFacility.va_regionid != null)
                    facility.RegionName = ppmsFacility.va_regionid.Name;
                if (ppmsFacility.va_visnid != null)
                    facility.VisnName = ppmsFacility.va_visnid.Name;

           
                //Facility Type Switch. 
                if (ppmsFacility.va_facilitytype != null)
                    switch (ppmsFacility.va_facilitytype.Value)
                    {
                        case (int)va_facility_va_facilitytype.VAMC:
                            facility.FacilityType = FacilityType.VAMC;
                            break;
                        case (int)va_facility_va_facilitytype.CLC:
                            facility.FacilityType = FacilityType.CLC;
                            break;
                                          
                    }

                //Extract the Related type from queryType string
                if (mapRelatedType.Contains("?$expand=Facilities"))
                    mapRelatedType = "Facilities";
                if (mapRelatedType.Contains("?$expand=CareSites"))
                    mapRelatedType = "CareSites";
                if (mapRelatedType.Contains("?$expand=Providers"))
                    mapRelatedType = "Providers";
                if (mapRelatedType.Contains("?$expand=Visn"))
                    mapRelatedType = "Visn";
                if (mapRelatedType.Contains("?$expand=ParentStation"))
                    mapRelatedType = "ParentStation";

                //Map Related Types Switch
                switch (mapRelatedType)
                {
                    case "Facilities":
                        //Lookup and Map the related Facilities
                        var ppmsRelatedFacilities =
                            context.va_facilitySet.Where(i => i.ppms_ParentStation.Id == ppmsFacility.Id);
                        var ppmsRelatedFacilitiesList = ppmsRelatedFacilities.ToList();
                        facility.Facilities =
                            await MapFacilities(ppmsRelatedFacilitiesList, "None", context);
                        break;
                    case "CareSites":
                        //Lookup and Map the Provider Agreements
                        var ppmsCareSites =
                            context.ppms_caresiteSet.Where(i => i.ppms_facilityid.Id == ppmsFacility.Id);
                        var ppmsCareSitesList = ppmsCareSites.ToList();
                        facility.CareSites =
                            await CareSiteMap.MapCareSites(ppmsCareSitesList, "None", context);
                        break;
                    case "Providers":
                        //Lookup and Map the Provider Agreements
                        var ppmsProviders =
                            context.AccountSet.Where(i => i.ppms_facilityid.Id == ppmsFacility.Id);
                        var ppmsProvidersList = ppmsProviders.ToList();
                        facility.Providers =
                            await ProviderMap.MapProviders(ppmsProvidersList, "None");
                        break;
                    case "Visn":
                        //Lookup and Map the Single Visn
                        var ppmsVisn =
                            context.va_visnSet.FirstOrDefault(i => i.Id == ppmsFacility.va_visnid.Id);
                        var ppmsVisnList = new List<va_visn> { ppmsVisn };
                        facility.Visn =
                            await VisnMap.MapVisns(ppmsVisnList, "None", context);
                        break;
                    case "ParentStation":
                        //Lookup and Map the Single Parent Station(facility)
                        var ppmsParentStation =
                            context.va_facilitySet.FirstOrDefault(i => i.Id == ppmsFacility.ppms_ParentStation.Id);
                        var ppmsParentStationList = new List<va_facility> { ppmsParentStation };
                        facility.ParentStation =
                            await FacilityMap.MapFacilities(ppmsParentStationList, "None", context);
                        break;

                    default:
                        break;
                }


                facilities.Add(facility);
            }
            return facilities.AsQueryable();
        }      
    }
}